<?php
/**
 * Plugin Name: Aurora Chat Widget (n8n) by BinariaOS
 * Description: Embeds the official n8n Chat widget on your WordPress site. Configure your webhook and theme the widget via CSS variables — no coding required.
 * Version: 1.0.0
 * Author: BinariaOS
 * Author URI: https://binariaos.com.py
 * Plugin URI: https://binariaos.com.py
 * Requires at least: 5.4
 * Tested up to: 6.6
 * Requires PHP: 7.4
 * Text Domain: aurora-chat-widget
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */
if ( ! defined('ABSPATH') ) exit;

define('ACW_VERSION', '1.0.0');
define('ACW_SLUG', 'aurora-chat-widget');
define('ACW_DIR', plugin_dir_path(__FILE__));
define('ACW_URL', plugin_dir_url(__FILE__));
define('ACW_UPDATE_URL', 'https://binariaos.com.py/plugins/aurora-chat/update.json');

function acw_load_textdomain() {
    load_plugin_textdomain('aurora-chat-widget', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('plugins_loaded', 'acw_load_textdomain');

function acw_default_options() {
    return array(
        'webhook'         => '',
        'mode'            => 'overlay',
        'position'        => 'bottom-right',
        'auto_load'       => 1,
        'inline_selector' => '#acw-chat',
        'default_lang'    => 'en',
        'initial_messages'=> "Hello! I'm your assistant.\nHow can I help today?",
        'i18n' => array(
            'title'            => 'Assistant',
            'subtitle'         => 'Chat Support',
            'inputPlaceholder' => 'Type your message…',
            'getStarted'       => 'Start chat',
            'typing'           => 'Assistant is typing…',
        ),
        'css_vars' => ":root {\n\t--chat--color--primary: #0ea5e9;\n\t--chat--color--primary-shade-50: #0891b2;\n\t--chat--color--primary--shade-100: #0369a1;\n\t--chat--color--secondary: #10b981;\n\t--chat--color-secondary-shade-50: #059669;\n\t--chat--color-white: #ffffff;\n\t--chat--color-light: #f2f4f8;\n\t--chat--color-light-shade-50: #e6e9f1;\n\t--chat--color-light-shade-100: #c2c5cc;\n\t--chat--color-medium: #d2d4d9;\n\t--chat--color-dark: #101330;\n\t--chat--color-disabled: #777980;\n\t--chat--color-typing: #404040;\n\n\t--chat--spacing: 1rem;\n\t--chat--border-radius: 0.75rem;\n\t--chat--transition-duration: 0.15s;\n\n\t--chat--window--width: 420px;\n\t--chat--window--height: 600px;\n\n\t--chat--header-height: auto;\n\t--chat--header--padding: var(--chat--spacing);\n\t--chat--header--background: var(--chat--color-dark);\n\t--chat--header--color: var(--chat--color-light);\n\t--chat--heading--font-size: 1.1rem;\n\t--chat--subtitle--font-size: 0.85rem;\n\t--chat--subtitle--line-height: 1.6;\n\n\t--chat--textarea--height: 56px;\n\n\t--chat--message--font-size: 1rem;\n\t--chat--message--padding: var(--chat--spacing);\n\t--chat--message--border-radius: var(--chat--border-radius);\n\t--chat--message-line-height: 1.7;\n\t--chat--message--bot--background: var(--chat--color-white);\n\t--chat--message--bot--color: var(--chat--color-dark);\n\t--chat--message--bot--border: none;\n\t--chat--message--user--background: var(--chat--color--secondary);\n\t--chat--message--user--color: var(--chat--color-white);\n\t--chat--message--user--border: none;\n\t--chat--message--pre--background: rgba(0,0,0,0.05);\n\n\t--chat--toggle--background: var(--chat--color--primary);\n\t--chat--toggle--hover--background: var(--chat--color--primary-shade-50);\n\t--chat--toggle--active--background: var(--chat--color--primary--shade-100);\n\t--chat--toggle--color: var(--chat--color-white);\n\t--chat--toggle--size: 64px;\n}",
    );
}

function acw_activate() {
    if ( ! get_option('acw_settings') ) {
        add_option('acw_settings', acw_default_options());
    }
}
register_activation_hook(__FILE__, 'acw_activate');

require_once ACW_DIR . 'includes/settings.php';
require_once ACW_DIR . 'admin/admin-screen.php';

function acw_shortcode($atts = array()) {
    $o = acw_get_settings();
    $o['mode'] = 'inline';
    return acw_render_chat($o, true);
}
add_shortcode('aurora_chat', 'acw_shortcode');

function acw_maybe_render_footer() {
    if (is_admin()) return;
    $o = acw_get_settings();
    if ( ! empty($o['auto_load']) ) {
        echo acw_render_chat($o, false);
    }
}
add_action('wp_footer', 'acw_maybe_render_footer', 100);

add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links){
  array_unshift($links, '<a href="'.admin_url('admin.php?page=acw-admin').'">'.__('Settings', 'aurora-chat-widget').'</a>');
  return $links;
});

add_filter('site_transient_update_plugins', function($transient){
  if ( empty($transient->checked) ) return $transient;
  $remote = get_transient('acw_update_manifest');
  if (!$remote) {
    $res = wp_remote_get(ACW_UPDATE_URL, array('timeout'=>5));
    if (!is_wp_error($res) && wp_remote_retrieve_response_code($res) === 200) {
      $remote = json_decode(wp_remote_retrieve_body($res));
      set_transient('acw_update_manifest', $remote, HOUR_IN_SECONDS);
    } else {
      return $transient;
    }
  }
  if (!empty($remote->new_version) && version_compare('1.0.0', $remote->new_version, '<')) {
    $plugin = plugin_basename(__FILE__);
    $transient->response[$plugin] = (object) array(
      'slug'        => ACW_SLUG,
      'new_version' => $remote->new_version,
      'url'         => $remote->url ?? '',
      'package'     => $remote->package ?? '',
    );
  }
  return $transient;
});

add_filter('plugins_api', function($result, $action, $args){
  if ($action !== 'plugin_information' || ($args->slug ?? '') !== ACW_SLUG) return $result;
  $remote = get_transient('acw_update_manifest');
  if (!$remote) {
    $res = wp_remote_get(ACW_UPDATE_URL, array('timeout'=>5));
    if (!is_wp_error($res) && wp_remote_retrieve_response_code($res) === 200) {
      $remote = json_decode(wp_remote_retrieve_body($res));
      set_transient('acw_update_manifest', $remote, HOUR_IN_SECONDS);
    }
  }
  if (!$remote) return $result;
  return (object) array(
    'name'          => 'Aurora Chat Widget (n8n) by BinariaOS',
    'slug'          => ACW_SLUG,
    'version'       => $remote->new_version ?? '1.0.0',
    'tested'        => $remote->tested ?? '6.6',
    'requires'      => $remote->requires ?? '5.4',
    'author'        => '<a href="https://binariaos.com.py">BinariaOS</a>',
    'homepage'      => $remote->url ?? 'https://binariaos.com.py',
    'download_link' => $remote->package ?? '',
    'sections'      => array(
      'description' => $remote->description ?? 'Lightweight WordPress wrapper for the official n8n chat widget.',
      'installation'=> '1) Upload and activate. 2) Go to Aurora Chat. 3) Paste your webhook. 4) Choose mode & theme.',
      'faq'         => 'Q: Does it host n8n locally? A: No, we load from jsDelivr.\nQ: Can I style it? A: Yes, paste CSS variables in admin.\nQ: How to use inline? A: Add a target DIV and set the selector.',
      'changelog'   => '* 1.0.0: First public release by BinariaOS.',
    ),
    'banners' => array(
      'low'  => ACW_URL . 'assets/banner-772x250.png',
      'high' => ACW_URL . 'assets/banner-772x250.png',
    ),
    'icons' => array(
      '1x' => ACW_URL . 'assets/icon-128x128.png',
    )
  );
}, 10, 3);

function acw_render_chat($opts, $return = false) {
    $webhook = isset($opts['webhook']) ? esc_url($opts['webhook']) : '';
    if (empty($webhook)) return '';

    $mode         = in_array(($opts['mode'] ?? 'overlay'), array('overlay','inline'), true) ? $opts['mode'] : 'overlay';
    $position     = $opts['position'] ?? 'bottom-right';
    $inline_sel   = $opts['inline_selector'] ?? '#acw-chat';
    $default_lang = $opts['default_lang'] ?? 'en';
    $initial      = trim((string)($opts['initial_messages'] ?? ''));
    $initial_js   = json_encode(array_filter(array_map('trim', explode("\n", $initial))));

    $i18n = is_array($opts['i18n'] ?? null) ? $opts['i18n'] : array();
    $i18n = array_merge(array(
        'title'            => 'Assistant',
        'subtitle'         => 'Chat Support',
        'inputPlaceholder' => 'Type your message…',
        'getStarted'       => 'Start chat',
        'typing'           => 'Assistant is typing…',
    ), $i18n);

    $css_vars = (string)($opts['css_vars'] ?? '');

    ob_start();
    ?>
    <!-- Aurora Chat Widget (n8n) by BinariaOS -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@n8n/chat/dist/style.css" />
    <?php if (!empty($css_vars)) : ?>
    <style><?php echo $css_vars; ?></style>
    <?php endif; ?>

    <?php if ($mode === 'inline') : ?>
      <div id="<?php echo esc_attr(ltrim($inline_sel, '#')); ?>" class="acw-inline-target"></div>
    <?php endif; ?>

    <script type="module">
      import { createChat } from "https://cdn.jsdelivr.net/npm/@n8n/chat/dist/chat.bundle.es.js";
      const options = {
        webhookUrl: <?php echo json_encode($webhook); ?>,
        defaultLanguage: <?php echo json_encode($default_lang); ?>,
        mode: <?php echo json_encode($mode); ?>,
        position: <?php echo json_encode($position); ?>,
        loadPreviousSession: true,
        initialMessages: <?php echo $initial_js; ?>,
        i18n: { <?php echo json_encode($default_lang); ?>: <?php echo json_encode($i18n); ?> }
        <?php if ($mode === 'inline') : ?>, target: <?php echo json_encode($inline_sel); ?><?php endif; ?>
      };
      const styleFix = document.createElement('style');
      styleFix.textContent = `
        .n8n-chat-widget [data-testid*="messages"],
        .n8n-chat-widget [class*="messages"] {
          overflow-y: auto !important;
          overflow-x: hidden !important;
          -webkit-overflow-scrolling: touch !important;
          touch-action: pan-y !important;
          scrollbar-width: thin;
        }
      `;
      document.head.appendChild(styleFix);
      createChat(options);
    </script>
    <?php
    return $return ? ob_get_clean() : print ob_get_clean();
}
